/*
Collision Mask Utilities
Copyright ©2014 David Powell

Released under the MIT Licence

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include "mainwindow.h"
#include "ui_mainwindow.h"

const QString ApplicationName("Collision Mask Creator");
const QString ApplicationVersion("V1.0");

/*!
 * \brief MainWindow constructor
 * \param parent The parent widget
 *
 * Initialise the application
 */
MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    ui->setupUi(this);

    qApp->installEventFilter(this);

    ui->statusBar->showMessage(tr("Ready"));
}

/*!
 * \brief MainWindow destructor
 */
MainWindow::~MainWindow()
{
    delete ui;
}

/*!
 * \brief Suppress changing the status tip automatically
 * \param event The event to check
 * \return true if the event should be filtered
 */
bool MainWindow::eventFilter(QObject * /* watched */, QEvent *event)
{
    return event->type() == QEvent::StatusTip;
}

/*!
 * \brief When the user drags an item; only allow files
 * \param event The drag event to check
 */
void MainWindow::dragEnterEvent(QDragEnterEvent *event)
{
    if (event->mimeData()->hasFormat("text/uri-list"))
    {
        event->acceptProposedAction();
    }
}

/*!
 * \brief When a file is dropped on the application;
 *        check it is a PNG file and then add it to the
 *        file list
 * \param event The drop event to use
 */
void MainWindow::dropEvent(QDropEvent *event)
{
    QList<QUrl> urls = event->mimeData()->urls();
    if (urls.isEmpty() == true)
    {
        return;
    }

    foreach (QUrl url, urls)
    {
        QString filename = url.toLocalFile();
        if (filename.isEmpty() == true)
        {
            continue;
        }

        QFileInfo fileInfo(filename);
        if (fileInfo.suffix().toLower() != "png")
        {
            continue;
        }

        QList<QListWidgetItem *> findResult =
                ui->fileListWidget->findItems(filename, Qt::MatchExactly);
        if (findResult.count() == 0)
        {
            ui->fileListWidget->addItem(filename);
        }
    }
}

/*!
 * \brief When the exit action is triggered; close the window
 *        thereby ending the application.
 */
void MainWindow::on_actionExit_triggered()
{
    this->close();
}

/*!
 * \brief When the about action is triggered;
 *        display the about box.
 */
void MainWindow::on_actionAbout_triggered()
{
    QMessageBox::information(this, ApplicationName,
                             ApplicationName + " " + ApplicationVersion + "\n" +
                             tr("Copyright ©2014 David Powell.\n") +
                             tr("Released under the MIT License.\n\n") +
                             tr("http://www.loadcode.co.uk"),
                             QMessageBox::Ok);
}

/*!
 * \brief When the check for updates action is triggered;
 *        Open the http://www.loadcode.co.uk URL
 */
void MainWindow::on_actionCheck_for_Updates_triggered()
{
     QDesktopServices::openUrl(QUrl("http://www.loadcode.co.uk"));
}

/*!
 * \brief When the add action is triggered;
 *        ask the user for a PNG file and then add
 *        it to the file list
 */
void MainWindow::on_actionAdd_triggered()
{
    QFileDialog dialog(this);
    dialog.setFileMode(QFileDialog::ExistingFiles);

    QStringList filters;
    filters << tr("PNG Image Files") + " (*.png)"
#ifdef Q_OS_WIN32
            << tr("All Files") + " (*.*)";
#else
            << tr("All Files") + " (*)";
#endif
    dialog.setNameFilters(filters);

    QStringList filenames;
    if (dialog.exec())
    {
        filenames = dialog.selectedFiles();
    }

    foreach (QString filename, filenames)
    {
        QList<QListWidgetItem *> findResult =
                ui->fileListWidget->findItems(filename, Qt::MatchExactly);
        if (findResult.count() == 0)
        {
            ui->fileListWidget->addItem(filename);
        }
    }
}

/*!
 * \brief When the remove action is triggered;
 *        remove all the currently select items in the file list
 */
void MainWindow::on_actionRemove_triggered()
{
    qDeleteAll(ui->fileListWidget->selectedItems());
}

/*!
 * \brief When the clear all action is triggered;
 *        clear the file list
 */
void MainWindow::on_actionClear_All_triggered()
{
    ui->fileListWidget->clear();
}

/*!
 * \brief When the go action is triggered;
 *        create the mask files from the images
 *        in the file list
 */
void MainWindow::on_actionGO_triggered()
{
    int fileCount = ui->fileListWidget->count();

    ui->progressBar->setValue(0);

    if (fileCount <= 0)
    {
        return;
    }

    ui->progressBar->setMaximum(fileCount);

    ui->statusBar->showMessage(tr("Processing"));

    for (int index = 0; index < fileCount; index++)
    {
        QString imageFilename = ui->fileListWidget->item(index)->text();
        QFileInfo imageFileInfo(imageFilename);
        QString imagePath = imageFileInfo.absolutePath();
        QString fileBase = imageFileInfo.completeBaseName();
        QString maskFilename(imagePath + QDir::separator() + fileBase + ".cmsk");

        createMaskFile(imageFilename, maskFilename);

        ui->progressBar->setValue(index + 1);
    }

    ui->statusBar->showMessage(tr("Ready"));
}

/*!
 * \brief Create a mask file from the specified image file
 * \param imageFilename The image file to use
 * \param maskFilename The mask file to create
 */
void MainWindow::createMaskFile(QString imageFilename, QString maskFilename)
{
    QImage image;
    bool loadResult = image.load(imageFilename, "PNG");

    if (loadResult == false)
    {
        QMessageBox::critical(this, qApp->applicationName(),
                              tr("The file '") + imageFilename +
                              tr("' cannot be loaded and has been skipped."),
                              QMessageBox::Ok);
        return;
    }

    int trimSize = ui->trimSpinBox->value();

    if ((trimSize * 2 >= image.width()) || (trimSize * 2 >= image.height()))
    {
        QMessageBox::critical(this, qApp->applicationName(),
                              tr("The file '") + imageFilename +
                              tr("' is too small to be trimmed and has been skipped."),
                              QMessageBox::Ok);
        return;
    }

    QImage trimmedImage = image.copy(trimSize, trimSize, image.width() - trimSize * 2,
                                     image.height() - trimSize *2);

    QVarLengthArray<quint8> pixels;

    for (int y = 0; y < trimmedImage.height(); y++)
    {
        for (int x = 0; x < trimmedImage.width(); x++)
        {
            pixels.append((QColor::fromRgba(trimmedImage.pixel(x,y))).alpha());
        }
    }

    QVarLengthArray<quint8> compressedPixels = rleEncode(pixels);

    QFile maskFile(maskFilename);

    if (maskFile.open(QIODevice::WriteOnly) == false)
    {
        QMessageBox::critical(this, qApp->applicationName(),
                              tr("The file '") + maskFilename +
                              tr("' cannot be opened and has been skipped."),
                              QMessageBox::Ok);
        return;
    }

    const quint8 signature[] = { 139, 'C', 'M', 'S', 'K', 13, 10, 26, 10, 0 };

    quint8 version[4];
    qToBigEndian((quint32)1, &version[0]);

    quint8 dataOffset[4];
    qToBigEndian((quint32)30, &dataOffset[0]);

    quint8 dataLength[4];
    qToBigEndian((quint32)compressedPixels.count(), &dataLength[0]);

    quint8 width[4];
    qToBigEndian((quint32)trimmedImage.width(), &width[0]);

    quint8 height[4];
    qToBigEndian((quint32)trimmedImage.height(), &height[0]);

    QVarLengthArray<quint8> fileData;

    fileData.append(&signature[0], 10);
    fileData.append(&version[0], 4);
    fileData.append(&dataOffset[0], 4);
    fileData.append(&dataLength[0], 4);
    fileData.append(&width[0], 4);
    fileData.append(&height[0], 4);
    fileData.append(compressedPixels.constData(), compressedPixels.count());

    quint8 crc[4];
    quint32 calcCrc = (quint32)calculateCrc32(fileData);
    qToBigEndian(calcCrc, &crc[0]);
    fileData.append(&crc[0], 4);

    maskFile.write((const char *)fileData.constData(), fileData.count());

    maskFile.close();
}

/*!
 * \brief RLE encoded the supplied pixel data
 * \param pixels The pixel data to encoded
 * \return The RLE encoded data
 */
QVarLengthArray<quint8> MainWindow::rleEncode(QVarLengthArray<quint8> pixels)
{
    quint8 lastPixel;
    quint8 currentPixel;
    quint32 numberDuplicated;
    QVarLengthArray<quint8> compressedData;

    currentPixel = pixels[0];
    lastPixel = currentPixel;
    numberDuplicated = 1;

    int offset = 0;

    while (offset < pixels.count() - 1)
    {
        offset++;
        lastPixel = currentPixel;
        currentPixel = pixels[offset];

        if (currentPixel == lastPixel)
        {
            numberDuplicated++;
        }
        else
        {
            while (numberDuplicated > 255)
            {
                compressedData.append(lastPixel);
                compressedData.append(lastPixel);
                compressedData.append(255);
                numberDuplicated -= 255;
            }

            if (numberDuplicated > 1)
            {
                compressedData.append(lastPixel);
                compressedData.append(lastPixel);
                compressedData.append((quint8)numberDuplicated);
                numberDuplicated = 1;
                continue;
            }

            if (numberDuplicated == 1)
            {
                compressedData.append(lastPixel);
            }

            numberDuplicated = 1;
        }
    }

    lastPixel = currentPixel;

    while (numberDuplicated > 255)
    {
        compressedData.append(lastPixel);
        compressedData.append(lastPixel);
        compressedData.append(255);
        numberDuplicated -= 255;
    }

    if (numberDuplicated > 1)
    {
        compressedData.append(lastPixel);
        compressedData.append(lastPixel);
        compressedData.append((quint8)numberDuplicated);
    }

    if (numberDuplicated == 1)
    {
        compressedData.append(lastPixel);
    }

    return compressedData;
}

/*!
 * \brief Calculate the CRC-32 checksum for the supplied data
 * \param data The data to checksum
 * \param previousCrc32 The previous CRC-32 checksum,
 *        if calculating a checksum in chunks
 * \return The CRC-32 checksum
 */
quint32 MainWindow::calculateCrc32(QVarLengthArray<quint8> data, quint32 previousCrc32)
{
    const quint32 Polynomial = 0xEDB88320;
    int index = 0;
    quint32 crc = ~previousCrc32;

    while (index < data.count())
    {
        crc ^= data[index++];
        for (quint32 bit = 0; bit < 8; bit++)
        {
            crc = (crc >> 1) ^ (-qint32(crc & 1) & Polynomial);
        }
    }

    return ~crc;
}
